/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import com.google.common.base.Preconditions;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.fs.HFileSystem;
import org.apache.hadoop.hbase.io.encoding.DataBlockEncoding;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.BoundedRangeFileInputStream;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializer;
import org.apache.hadoop.hbase.io.hfile.ChecksumUtil;
import org.apache.hadoop.hbase.io.hfile.Compression;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFileDataBlockEncoder;
import org.apache.hadoop.hbase.io.hfile.NoOpDataBlockEncoder;
import org.apache.hadoop.hbase.regionserver.metrics.SchemaConfigured;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.ChecksumType;
import org.apache.hadoop.hbase.util.ClassSize;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.util.Writables;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.io.compress.CompressionOutputStream;
import org.apache.hadoop.io.compress.Compressor;
import org.apache.hadoop.io.compress.Decompressor;

public class HFileBlock
extends SchemaConfigured
implements Cacheable {
    static final int MINOR_VERSION_WITH_CHECKSUM = 1;
    static final int MINOR_VERSION_NO_CHECKSUM = 0;
    static final int CHECKSUM_VERIFICATION_NUM_IO_THRESHOLD = 3;
    static final int HEADER_SIZE_NO_CHECKSUM = 24;
    public static final boolean FILL_HEADER = true;
    public static final boolean DONT_FILL_HEADER = false;
    static final int HEADER_SIZE_WITH_CHECKSUMS = 33;
    public static final int ENCODED_HEADER_SIZE = 35;
    static final byte[] DUMMY_HEADER_WITH_CHECKSUM = new byte[33];
    static final byte[] DUMMY_HEADER_NO_CHECKSUM = new byte[24];
    public static final int BYTE_BUFFER_HEAP_SIZE = (int)ClassSize.estimateBase(ByteBuffer.wrap(new byte[0], 0, 0).getClass(), false);
    static final int EXTRA_SERIALIZATION_SPACE = 12;
    static final int CHECKSUM_SIZE = 4;
    private static final CacheableDeserializer<Cacheable> blockDeserializer = new CacheableDeserializer<Cacheable>(){

        @Override
        public HFileBlock deserialize(ByteBuffer buf) throws IOException {
            ByteBuffer newByteBuffer = ByteBuffer.allocate(buf.limit() - 12);
            buf.limit(buf.limit() - 12).rewind();
            newByteBuffer.put(buf);
            HFileBlock ourBuffer = new HFileBlock(newByteBuffer, 0);
            buf.position(buf.limit());
            buf.limit(buf.limit() + 12);
            ourBuffer.offset = buf.getLong();
            ourBuffer.nextBlockOnDiskSizeWithHeader = buf.getInt();
            return ourBuffer;
        }
    };
    private BlockType blockType;
    private int onDiskSizeWithoutHeader;
    private final int uncompressedSizeWithoutHeader;
    private final long prevBlockOffset;
    private final byte checksumType;
    private final int bytesPerChecksum;
    private final int onDiskDataSizeWithHeader;
    private final int minorVersion;
    private ByteBuffer buf;
    private boolean includesMemstoreTS;
    private long offset = -1L;
    private int nextBlockOnDiskSizeWithHeader = -1;

    HFileBlock(BlockType blockType, int onDiskSizeWithoutHeader, int uncompressedSizeWithoutHeader, long prevBlockOffset, ByteBuffer buf, boolean fillHeader, long offset, boolean includesMemstoreTS, int minorVersion, int bytesPerChecksum, byte checksumType, int onDiskDataSizeWithHeader) {
        this.blockType = blockType;
        this.onDiskSizeWithoutHeader = onDiskSizeWithoutHeader;
        this.uncompressedSizeWithoutHeader = uncompressedSizeWithoutHeader;
        this.prevBlockOffset = prevBlockOffset;
        this.buf = buf;
        if (fillHeader) {
            this.overwriteHeader();
        }
        this.offset = offset;
        this.includesMemstoreTS = includesMemstoreTS;
        this.minorVersion = minorVersion;
        this.bytesPerChecksum = bytesPerChecksum;
        this.checksumType = checksumType;
        this.onDiskDataSizeWithHeader = onDiskDataSizeWithHeader;
    }

    HFileBlock(ByteBuffer b, int minorVersion) throws IOException {
        b.rewind();
        this.blockType = BlockType.read(b);
        this.onDiskSizeWithoutHeader = b.getInt();
        this.uncompressedSizeWithoutHeader = b.getInt();
        this.prevBlockOffset = b.getLong();
        this.minorVersion = minorVersion;
        if (minorVersion >= 1) {
            this.checksumType = b.get();
            this.bytesPerChecksum = b.getInt();
            this.onDiskDataSizeWithHeader = b.getInt();
        } else {
            this.checksumType = ChecksumType.NULL.getCode();
            this.bytesPerChecksum = 0;
            this.onDiskDataSizeWithHeader = this.onDiskSizeWithoutHeader + 24;
        }
        this.buf = b;
        this.buf.rewind();
    }

    @Override
    public BlockType getBlockType() {
        return this.blockType;
    }

    public short getDataBlockEncodingId() {
        if (this.blockType != BlockType.ENCODED_DATA) {
            throw new IllegalArgumentException("Querying encoder ID of a block of type other than " + (Object)((Object)BlockType.ENCODED_DATA) + ": " + (Object)((Object)this.blockType));
        }
        return this.buf.getShort(this.headerSize());
    }

    public int getOnDiskSizeWithHeader() {
        return this.onDiskSizeWithoutHeader + this.headerSize();
    }

    int getOnDiskSizeWithoutHeader() {
        return this.onDiskSizeWithoutHeader;
    }

    public int getUncompressedSizeWithoutHeader() {
        return this.uncompressedSizeWithoutHeader;
    }

    public long getPrevBlockOffset() {
        return this.prevBlockOffset;
    }

    private void overwriteHeader() {
        this.buf.rewind();
        this.blockType.write(this.buf);
        this.buf.putInt(this.onDiskSizeWithoutHeader);
        this.buf.putInt(this.uncompressedSizeWithoutHeader);
        this.buf.putLong(this.prevBlockOffset);
    }

    ByteBuffer getBufferWithoutHeader() {
        return ByteBuffer.wrap(this.buf.array(), this.buf.arrayOffset() + this.headerSize(), this.buf.limit() - this.headerSize() - this.totalChecksumBytes()).slice();
    }

    public ByteBuffer getBufferReadOnly() {
        return ByteBuffer.wrap(this.buf.array(), this.buf.arrayOffset(), this.buf.limit() - this.totalChecksumBytes()).slice();
    }

    ByteBuffer getBufferWithHeader() {
        ByteBuffer dupBuf = this.buf.duplicate();
        dupBuf.rewind();
        return dupBuf;
    }

    void readInto(Writable w) throws IOException {
        Preconditions.checkNotNull((Object)w);
        if (Writables.getWritable(this.buf.array(), this.buf.arrayOffset() + this.headerSize(), this.buf.limit() - this.headerSize(), w) == null) {
            throw new IOException("Failed to deserialize block " + this + " into a " + w.getClass().getSimpleName());
        }
    }

    private void sanityCheckAssertion(long valueFromBuf, long valueFromField, String fieldName) throws IOException {
        if (valueFromBuf != valueFromField) {
            throw new AssertionError((Object)(fieldName + " in the buffer (" + valueFromBuf + ") is different from that in the field (" + valueFromField + ")"));
        }
    }

    void sanityCheck() throws IOException {
        this.buf.rewind();
        BlockType blockTypeFromBuf = BlockType.read(this.buf);
        if (blockTypeFromBuf != this.blockType) {
            throw new IOException("Block type stored in the buffer: " + (Object)((Object)blockTypeFromBuf) + ", block type field: " + (Object)((Object)this.blockType));
        }
        this.sanityCheckAssertion(this.buf.getInt(), this.onDiskSizeWithoutHeader, "onDiskSizeWithoutHeader");
        this.sanityCheckAssertion(this.buf.getInt(), this.uncompressedSizeWithoutHeader, "uncompressedSizeWithoutHeader");
        this.sanityCheckAssertion(this.buf.getLong(), this.prevBlockOffset, "prevBlocKOffset");
        if (this.minorVersion >= 1) {
            this.sanityCheckAssertion(this.buf.get(), this.checksumType, "checksumType");
            this.sanityCheckAssertion(this.buf.getInt(), this.bytesPerChecksum, "bytesPerChecksum");
            this.sanityCheckAssertion(this.buf.getInt(), this.onDiskDataSizeWithHeader, "onDiskDataSizeWithHeader");
        }
        int cksumBytes = this.totalChecksumBytes();
        int hdrSize = this.headerSize();
        int expectedBufLimit = this.uncompressedSizeWithoutHeader + this.headerSize() + cksumBytes;
        if (this.buf.limit() != expectedBufLimit) {
            throw new AssertionError((Object)("Expected buffer limit " + expectedBufLimit + ", got " + this.buf.limit()));
        }
        int size = this.uncompressedSizeWithoutHeader + hdrSize + cksumBytes;
        if (this.buf.capacity() != size && this.buf.capacity() != size + hdrSize) {
            throw new AssertionError((Object)("Invalid buffer capacity: " + this.buf.capacity() + ", expected " + size + " or " + (size + hdrSize)));
        }
    }

    public String toString() {
        return "blockType=" + (Object)((Object)this.blockType) + ", onDiskSizeWithoutHeader=" + this.onDiskSizeWithoutHeader + ", uncompressedSizeWithoutHeader=" + this.uncompressedSizeWithoutHeader + ", prevBlockOffset=" + this.prevBlockOffset + ", dataBeginsWith=" + Bytes.toStringBinary(this.buf.array(), this.buf.arrayOffset() + this.headerSize(), Math.min(32, this.buf.limit() - this.buf.arrayOffset() - this.headerSize())) + ", fileOffset=" + this.offset;
    }

    private void validateOnDiskSizeWithoutHeader(int expectedOnDiskSizeWithoutHeader) throws IOException {
        if (this.onDiskSizeWithoutHeader != expectedOnDiskSizeWithoutHeader) {
            String blockInfoMsg = "Block offset: " + this.offset + ", data starts with: " + Bytes.toStringBinary(this.buf.array(), this.buf.arrayOffset(), this.buf.arrayOffset() + Math.min(32, this.buf.limit()));
            throw new IOException("On-disk size without header provided is " + expectedOnDiskSizeWithoutHeader + ", but block " + "header contains " + this.onDiskSizeWithoutHeader + ". " + blockInfoMsg);
        }
    }

    private void allocateBuffer(boolean extraBytes) {
        int cksumBytes = this.totalChecksumBytes();
        int capacityNeeded = this.headerSize() + this.uncompressedSizeWithoutHeader + cksumBytes + (extraBytes ? this.headerSize() : 0);
        ByteBuffer newBuf = ByteBuffer.allocate(capacityNeeded);
        System.arraycopy(this.buf.array(), this.buf.arrayOffset(), newBuf.array(), newBuf.arrayOffset(), this.headerSize());
        this.buf = newBuf;
        this.buf.limit(this.headerSize() + this.uncompressedSizeWithoutHeader + cksumBytes);
    }

    public void assumeUncompressed() throws IOException {
        if (this.onDiskSizeWithoutHeader != this.uncompressedSizeWithoutHeader + this.totalChecksumBytes()) {
            throw new IOException("Using no compression but onDiskSizeWithoutHeader=" + this.onDiskSizeWithoutHeader + ", " + "uncompressedSizeWithoutHeader=" + this.uncompressedSizeWithoutHeader + ", numChecksumbytes=" + this.totalChecksumBytes());
        }
    }

    public void expectType(BlockType expectedType) throws IOException {
        if (this.blockType != expectedType) {
            throw new IOException("Invalid block type: expected=" + (Object)((Object)expectedType) + ", actual=" + (Object)((Object)this.blockType));
        }
    }

    public long getOffset() {
        if (this.offset < 0L) {
            throw new IllegalStateException("HFile block offset not initialized properly");
        }
        return this.offset;
    }

    public DataInputStream getByteStream() {
        return new DataInputStream(new ByteArrayInputStream(this.buf.array(), this.buf.arrayOffset() + this.headerSize(), this.buf.limit() - this.headerSize()));
    }

    @Override
    public long heapSize() {
        long size = ClassSize.align(SCHEMA_CONFIGURED_UNALIGNED_HEAP_SIZE + 2 * ClassSize.REFERENCE + 24 + 1 + 16 + 1);
        if (this.buf != null) {
            size += (long)ClassSize.align(this.buf.capacity() + BYTE_BUFFER_HEAP_SIZE);
        }
        return ClassSize.align(size);
    }

    public static boolean readWithExtra(InputStream in, byte[] buf, int bufOffset, int necessaryLen, int extraLen) throws IOException {
        int bytesRemaining;
        int ret;
        for (bytesRemaining = necessaryLen + extraLen; bytesRemaining > 0 && ((ret = in.read(buf, bufOffset, bytesRemaining)) != -1 || bytesRemaining > extraLen); bytesRemaining -= ret) {
            if (ret < 0) {
                throw new IOException("Premature EOF from inputStream (read returned " + ret + ", was trying to read " + necessaryLen + " necessary bytes and " + extraLen + " extra bytes, " + "successfully read " + (necessaryLen + extraLen - bytesRemaining));
            }
            bufOffset += ret;
        }
        return bytesRemaining <= 0;
    }

    public int getNextBlockOnDiskSizeWithHeader() {
        return this.nextBlockOnDiskSizeWithHeader;
    }

    @Override
    public int getSerializedLength() {
        if (this.buf != null) {
            return this.buf.limit() + 12;
        }
        return 0;
    }

    @Override
    public void serialize(ByteBuffer destination) {
        destination.put(this.buf.duplicate());
        destination.putLong(this.offset);
        destination.putInt(this.nextBlockOnDiskSizeWithHeader);
        destination.rewind();
    }

    @Override
    public CacheableDeserializer<Cacheable> getDeserializer() {
        return blockDeserializer;
    }

    public boolean equals(Object comparison) {
        if (this == comparison) {
            return true;
        }
        if (comparison == null) {
            return false;
        }
        if (comparison.getClass() != this.getClass()) {
            return false;
        }
        HFileBlock castedComparison = (HFileBlock)comparison;
        if (castedComparison.blockType != this.blockType) {
            return false;
        }
        if (castedComparison.nextBlockOnDiskSizeWithHeader != this.nextBlockOnDiskSizeWithHeader) {
            return false;
        }
        if (castedComparison.offset != this.offset) {
            return false;
        }
        if (castedComparison.onDiskSizeWithoutHeader != this.onDiskSizeWithoutHeader) {
            return false;
        }
        if (castedComparison.prevBlockOffset != this.prevBlockOffset) {
            return false;
        }
        if (castedComparison.uncompressedSizeWithoutHeader != this.uncompressedSizeWithoutHeader) {
            return false;
        }
        if (this.buf.compareTo(castedComparison.buf) != 0) {
            return false;
        }
        if (this.buf.position() != castedComparison.buf.position()) {
            return false;
        }
        return this.buf.limit() == castedComparison.buf.limit();
    }

    public boolean doesIncludeMemstoreTS() {
        return this.includesMemstoreTS;
    }

    public DataBlockEncoding getDataBlockEncoding() {
        if (this.blockType == BlockType.ENCODED_DATA) {
            return DataBlockEncoding.getEncodingById(this.getDataBlockEncodingId());
        }
        return DataBlockEncoding.NONE;
    }

    byte getChecksumType() {
        return this.checksumType;
    }

    int getBytesPerChecksum() {
        return this.bytesPerChecksum;
    }

    int getOnDiskDataSizeWithHeader() {
        return this.onDiskDataSizeWithHeader;
    }

    int getMinorVersion() {
        return this.minorVersion;
    }

    int totalChecksumBytes() {
        if (this.minorVersion < 1 || this.bytesPerChecksum == 0) {
            return 0;
        }
        return (int)ChecksumUtil.numBytes(this.onDiskDataSizeWithHeader, this.bytesPerChecksum);
    }

    public int headerSize() {
        return HFileBlock.headerSize(this.minorVersion);
    }

    private static int headerSize(int minorVersion) {
        if (minorVersion < 1) {
            return 24;
        }
        return 33;
    }

    public byte[] getDummyHeaderForVersion() {
        return HFileBlock.getDummyHeaderForVersion(this.minorVersion);
    }

    private static byte[] getDummyHeaderForVersion(int minorVersion) {
        if (minorVersion < 1) {
            return DUMMY_HEADER_NO_CHECKSUM;
        }
        return DUMMY_HEADER_WITH_CHECKSUM;
    }

    static String toStringHeader(ByteBuffer buf) throws IOException {
        int offset = buf.arrayOffset();
        byte[] b = buf.array();
        long magic = Bytes.toLong(b, offset);
        BlockType bt = BlockType.read(buf);
        int compressedBlockSizeNoHeader = Bytes.toInt(b, offset += 8);
        int uncompressedBlockSizeNoHeader = Bytes.toInt(b, offset += 4);
        long prevBlockOffset = Bytes.toLong(b, offset += 4);
        byte cksumtype = b[offset += 8];
        long bytesPerChecksum = Bytes.toInt(b, ++offset);
        long onDiskDataSizeWithHeader = Bytes.toInt(b, offset += 4);
        offset += 4;
        return " Header dump: magic: " + magic + " blockType " + (Object)((Object)bt) + " compressedBlockSizeNoHeader " + compressedBlockSizeNoHeader + " uncompressedBlockSizeNoHeader " + uncompressedBlockSizeNoHeader + " prevBlockOffset " + prevBlockOffset + " checksumType " + (Object)((Object)ChecksumType.codeToType(cksumtype)) + " bytesPerChecksum " + bytesPerChecksum + " onDiskDataSizeWithHeader " + onDiskDataSizeWithHeader;
    }

    static class FSReaderV2
    extends AbstractFSReader {
        private final boolean useHBaseChecksumConfigured;
        private volatile boolean useHBaseChecksum;
        private volatile int checksumOffCount = -1;
        protected boolean includesMemstoreTS;
        protected HFileDataBlockEncoder dataBlockEncoder = NoOpDataBlockEncoder.INSTANCE;
        private ThreadLocal<PrefetchedHeader> prefetchedHeaderForThread = new ThreadLocal<PrefetchedHeader>(){

            @Override
            public PrefetchedHeader initialValue() {
                return new PrefetchedHeader();
            }
        };

        public FSReaderV2(FSDataInputStream istream, FSDataInputStream istreamNoFsChecksum, Compression.Algorithm compressAlgo, long fileSize, int minorVersion, HFileSystem hfs, Path path) throws IOException {
            super(istream, istreamNoFsChecksum, compressAlgo, fileSize, minorVersion, hfs, path);
            this.useHBaseChecksum = hfs != null ? hfs.useHBaseChecksum() : true;
            if (this.getMinorVersion() < 1) {
                this.useHBaseChecksum = false;
            }
            this.useHBaseChecksumConfigured = this.useHBaseChecksum;
        }

        FSReaderV2(FSDataInputStream istream, Compression.Algorithm compressAlgo, long fileSize) throws IOException {
            this(istream, istream, compressAlgo, fileSize, 1, null, null);
        }

        @Override
        public HFileBlock readBlockData(long offset, long onDiskSizeWithHeaderL, int uncompressedSize, boolean pread) throws IOException {
            HFileBlock blk;
            FSDataInputStream is = this.istreamNoFsChecksum;
            boolean doVerificationThruHBaseChecksum = this.useHBaseChecksum;
            if (!doVerificationThruHBaseChecksum) {
                is = this.istream;
            }
            if ((blk = this.readBlockDataInternal(is, offset, onDiskSizeWithHeaderL, uncompressedSize, pread, doVerificationThruHBaseChecksum)) == null) {
                HFile.LOG.warn((Object)("HBase checksum verification failed for file " + this.path + " at offset " + offset + " filesize " + this.fileSize + ". Retrying read with HDFS checksums turned on..."));
                if (!doVerificationThruHBaseChecksum) {
                    String msg = "HBase checksum verification failed for file " + this.path + " at offset " + offset + " filesize " + this.fileSize + " but this cannot happen because doVerify is " + doVerificationThruHBaseChecksum;
                    HFile.LOG.warn((Object)msg);
                    throw new IOException(msg);
                }
                HFile.checksumFailures.incrementAndGet();
                this.checksumOffCount = 3;
                this.useHBaseChecksum = false;
                doVerificationThruHBaseChecksum = false;
                is = this.istream;
                blk = this.readBlockDataInternal(is, offset, onDiskSizeWithHeaderL, uncompressedSize, pread, doVerificationThruHBaseChecksum);
                if (blk != null) {
                    HFile.LOG.warn((Object)("HDFS checksum verification suceeded for file " + this.path + " at offset " + offset + " filesize " + this.fileSize));
                }
            }
            if (blk == null && !doVerificationThruHBaseChecksum) {
                String msg = "readBlockData failed, possibly due to checksum verification failed for file " + this.path + " at offset " + offset + " filesize " + this.fileSize;
                HFile.LOG.warn((Object)msg);
                throw new IOException(msg);
            }
            if (!this.useHBaseChecksum && this.useHBaseChecksumConfigured && this.checksumOffCount-- < 0) {
                this.useHBaseChecksum = true;
            }
            return blk;
        }

        private HFileBlock readBlockDataInternal(FSDataInputStream is, long offset, long onDiskSizeWithHeaderL, int uncompressedSize, boolean pread, boolean verifyChecksum) throws IOException {
            HFileBlock b;
            if (offset < 0L) {
                throw new IOException("Invalid offset=" + offset + " trying to read " + "block (onDiskSize=" + onDiskSizeWithHeaderL + ", uncompressedSize=" + uncompressedSize + ")");
            }
            if (uncompressedSize != -1) {
                throw new IOException("Version 2 block reader API does not need the uncompressed size parameter");
            }
            if (onDiskSizeWithHeaderL < (long)this.hdrSize && onDiskSizeWithHeaderL != -1L || onDiskSizeWithHeaderL >= Integer.MAX_VALUE) {
                throw new IOException("Invalid onDisksize=" + onDiskSizeWithHeaderL + ": expected to be at least " + this.hdrSize + " and at most " + Integer.MAX_VALUE + ", or -1 (offset=" + offset + ", uncompressedSize=" + uncompressedSize + ")");
            }
            int onDiskSizeWithHeader = (int)onDiskSizeWithHeaderL;
            if (onDiskSizeWithHeader > 0) {
                int preReadHeaderSize;
                int onDiskSizeWithoutHeader = onDiskSizeWithHeader - this.hdrSize;
                assert (onDiskSizeWithoutHeader >= 0);
                PrefetchedHeader prefetchedHeader = this.prefetchedHeaderForThread.get();
                byte[] header = prefetchedHeader.offset == offset ? prefetchedHeader.header : null;
                int n = preReadHeaderSize = header == null ? 0 : this.hdrSize;
                if (this.compressAlgo == Compression.Algorithm.NONE) {
                    ByteBuffer headerAndData = ByteBuffer.allocate(onDiskSizeWithHeader + this.hdrSize);
                    headerAndData.limit(onDiskSizeWithHeader);
                    if (header != null) {
                        System.arraycopy(header, 0, headerAndData.array(), 0, this.hdrSize);
                    }
                    int nextBlockOnDiskSizeWithHeader = this.readAtOffset(is, headerAndData.array(), headerAndData.arrayOffset() + preReadHeaderSize, onDiskSizeWithHeader - preReadHeaderSize, true, offset + (long)preReadHeaderSize, pread);
                    b = new HFileBlock(headerAndData, this.getMinorVersion());
                    b.assumeUncompressed();
                    b.validateOnDiskSizeWithoutHeader(onDiskSizeWithoutHeader);
                    b.nextBlockOnDiskSizeWithHeader = nextBlockOnDiskSizeWithHeader;
                    if (verifyChecksum && !this.validateBlockChecksum(b, headerAndData.array(), this.hdrSize)) {
                        return null;
                    }
                    if (b.nextBlockOnDiskSizeWithHeader > 0) {
                        this.setNextBlockHeader(offset, b);
                    }
                } else {
                    byte[] onDiskBlock = new byte[onDiskSizeWithHeader + this.hdrSize];
                    int nextBlockOnDiskSize = this.readAtOffset(is, onDiskBlock, preReadHeaderSize, onDiskSizeWithHeader - preReadHeaderSize, true, offset + (long)preReadHeaderSize, pread);
                    if (header == null) {
                        header = onDiskBlock;
                    }
                    try {
                        b = new HFileBlock(ByteBuffer.wrap(header, 0, this.hdrSize), this.getMinorVersion());
                    }
                    catch (IOException ex) {
                        throw new IOException("Failed to read compressed block at " + offset + ", onDiskSizeWithoutHeader=" + onDiskSizeWithHeader + ", preReadHeaderSize=" + preReadHeaderSize + ", header.length=" + header.length + ", header bytes: " + Bytes.toStringBinary(header, 0, this.hdrSize), ex);
                    }
                    b.validateOnDiskSizeWithoutHeader(onDiskSizeWithoutHeader);
                    b.nextBlockOnDiskSizeWithHeader = nextBlockOnDiskSize;
                    if (verifyChecksum && !this.validateBlockChecksum(b, onDiskBlock, this.hdrSize)) {
                        return null;
                    }
                    DataInputStream dis = new DataInputStream(new ByteArrayInputStream(onDiskBlock, this.hdrSize, onDiskSizeWithoutHeader));
                    b.allocateBuffer(b.nextBlockOnDiskSizeWithHeader > 0);
                    this.decompress(b.buf.array(), b.buf.arrayOffset() + this.hdrSize, dis, b.uncompressedSizeWithoutHeader);
                    if (nextBlockOnDiskSize > 0) {
                        System.arraycopy(onDiskBlock, onDiskSizeWithHeader, b.buf.array(), b.buf.arrayOffset() + this.hdrSize + b.uncompressedSizeWithoutHeader + b.totalChecksumBytes(), this.hdrSize);
                        this.setNextBlockHeader(offset, b);
                    }
                }
            } else {
                ByteBuffer headerBuf;
                PrefetchedHeader prefetchedHeader = this.prefetchedHeaderForThread.get();
                ByteBuffer byteBuffer = headerBuf = prefetchedHeader.offset == offset ? prefetchedHeader.buf : null;
                if (headerBuf == null) {
                    headerBuf = ByteBuffer.allocate(this.hdrSize);
                    this.readAtOffset(is, headerBuf.array(), headerBuf.arrayOffset(), this.hdrSize, false, offset, pread);
                }
                b = new HFileBlock(headerBuf, this.getMinorVersion());
                b.allocateBuffer(true);
                if (this.compressAlgo == Compression.Algorithm.NONE) {
                    b.assumeUncompressed();
                    b.nextBlockOnDiskSizeWithHeader = this.readAtOffset(is, b.buf.array(), b.buf.arrayOffset() + this.hdrSize, b.uncompressedSizeWithoutHeader + b.totalChecksumBytes(), true, offset + (long)this.hdrSize, pread);
                    if (verifyChecksum && !this.validateBlockChecksum(b, b.buf.array(), this.hdrSize)) {
                        return null;
                    }
                    if (b.nextBlockOnDiskSizeWithHeader > 0) {
                        this.setNextBlockHeader(offset, b);
                    }
                } else {
                    byte[] compressedBytes = new byte[b.getOnDiskSizeWithHeader() + this.hdrSize];
                    b.nextBlockOnDiskSizeWithHeader = this.readAtOffset(is, compressedBytes, this.hdrSize, b.onDiskSizeWithoutHeader, true, offset + (long)this.hdrSize, pread);
                    if (verifyChecksum && !this.validateBlockChecksum(b, compressedBytes, this.hdrSize)) {
                        return null;
                    }
                    DataInputStream dis = new DataInputStream(new ByteArrayInputStream(compressedBytes, this.hdrSize, b.onDiskSizeWithoutHeader));
                    this.decompress(b.buf.array(), b.buf.arrayOffset() + this.hdrSize, dis, b.uncompressedSizeWithoutHeader);
                    if (b.nextBlockOnDiskSizeWithHeader > 0) {
                        int nextHeaderOffset = b.buf.arrayOffset() + this.hdrSize + b.uncompressedSizeWithoutHeader + b.totalChecksumBytes();
                        System.arraycopy(compressedBytes, compressedBytes.length - this.hdrSize, b.buf.array(), nextHeaderOffset, this.hdrSize);
                        this.setNextBlockHeader(offset, b);
                    }
                }
            }
            b.includesMemstoreTS = this.includesMemstoreTS;
            b.offset = offset;
            return b;
        }

        private void setNextBlockHeader(long offset, HFileBlock b) {
            PrefetchedHeader prefetchedHeader = this.prefetchedHeaderForThread.get();
            prefetchedHeader.offset = offset + (long)b.getOnDiskSizeWithHeader();
            int nextHeaderOffset = b.buf.arrayOffset() + this.hdrSize + b.uncompressedSizeWithoutHeader + b.totalChecksumBytes();
            System.arraycopy(b.buf.array(), nextHeaderOffset, prefetchedHeader.header, 0, this.hdrSize);
        }

        void setIncludesMemstoreTS(boolean enabled) {
            this.includesMemstoreTS = enabled;
        }

        void setDataBlockEncoder(HFileDataBlockEncoder encoder) {
            this.dataBlockEncoder = encoder;
        }

        protected boolean validateBlockChecksum(HFileBlock block, byte[] data, int hdrSize) throws IOException {
            return ChecksumUtil.validateBlockChecksum(this.path, block, data, hdrSize);
        }
    }

    private static class PrefetchedHeader {
        long offset = -1L;
        byte[] header = new byte[33];
        ByteBuffer buf = ByteBuffer.wrap(this.header, 0, 33);

        private PrefetchedHeader() {
        }
    }

    static class FSReaderV1
    extends AbstractFSReader {
        private static final int HEADER_DELTA = 16;

        public FSReaderV1(FSDataInputStream istream, Compression.Algorithm compressAlgo, long fileSize) throws IOException {
            super(istream, istream, compressAlgo, fileSize, 0, null, null);
        }

        @Override
        public HFileBlock readBlockData(long offset, long onDiskSizeWithMagic, int uncompressedSizeWithMagic, boolean pread) throws IOException {
            int onDiskSizeWithoutHeader;
            if (uncompressedSizeWithMagic <= 0) {
                throw new IOException("Invalid uncompressedSize=" + uncompressedSizeWithMagic + " for a version 1 block");
            }
            if (onDiskSizeWithMagic <= 0L || onDiskSizeWithMagic >= Integer.MAX_VALUE) {
                throw new IOException("Invalid onDiskSize=" + onDiskSizeWithMagic + " (maximum allowed: " + Integer.MAX_VALUE + ")");
            }
            int onDiskSize = (int)onDiskSizeWithMagic;
            if (uncompressedSizeWithMagic < 8) {
                throw new IOException("Uncompressed size for a version 1 block is " + uncompressedSizeWithMagic + " but must be at least " + 8);
            }
            ByteBuffer buf = ByteBuffer.allocate(uncompressedSizeWithMagic + 16);
            if (this.compressAlgo == Compression.Algorithm.NONE) {
                if (onDiskSize != uncompressedSizeWithMagic) {
                    throw new IOException("onDiskSize=" + onDiskSize + " and uncompressedSize=" + uncompressedSizeWithMagic + " must be equal for version 1 with no compression");
                }
                this.readAtOffset(this.istream, buf.array(), buf.arrayOffset() + 16, onDiskSize, false, offset, pread);
                onDiskSizeWithoutHeader = uncompressedSizeWithMagic - 8;
            } else {
                InputStream bufferedBoundedStream = this.createBufferedBoundedStream(offset, onDiskSize, pread);
                this.decompress(buf.array(), buf.arrayOffset() + 16, bufferedBoundedStream, uncompressedSizeWithMagic);
                onDiskSizeWithoutHeader = onDiskSize;
            }
            BlockType newBlockType = BlockType.parse(buf.array(), buf.arrayOffset() + 16, 8);
            HFileBlock b = new HFileBlock(newBlockType, onDiskSizeWithoutHeader, uncompressedSizeWithMagic - 8, -1L, buf, true, offset, false, 0, 0, ChecksumType.NULL.getCode(), onDiskSizeWithoutHeader + 24);
            return b;
        }
    }

    private static abstract class AbstractFSReader
    implements FSReader {
        protected final FSDataInputStream istream;
        protected final FSDataInputStream istreamNoFsChecksum;
        protected Compression.Algorithm compressAlgo;
        protected long fileSize;
        private int minorVersion;
        protected int hdrSize;
        protected HFileSystem hfs;
        protected Path path;
        private final Lock streamLock = new ReentrantLock();
        public static final int DEFAULT_BUFFER_SIZE = 0x100000;

        public AbstractFSReader(FSDataInputStream istream, FSDataInputStream istreamNoFsChecksum, Compression.Algorithm compressAlgo, long fileSize, int minorVersion, HFileSystem hfs, Path path) throws IOException {
            this.istream = istream;
            this.compressAlgo = compressAlgo;
            this.fileSize = fileSize;
            this.minorVersion = minorVersion;
            this.hfs = hfs;
            this.path = path;
            this.hdrSize = HFileBlock.headerSize(minorVersion);
            this.istreamNoFsChecksum = istreamNoFsChecksum;
        }

        @Override
        public BlockIterator blockRange(final long startOffset, final long endOffset) {
            return new BlockIterator(){
                private long offset;
                {
                    this.offset = startOffset;
                }

                @Override
                public HFileBlock nextBlock() throws IOException {
                    if (this.offset >= endOffset) {
                        return null;
                    }
                    HFileBlock b = AbstractFSReader.this.readBlockData(this.offset, -1L, -1, false);
                    this.offset += (long)b.getOnDiskSizeWithHeader();
                    return b;
                }

                @Override
                public HFileBlock nextBlockWithBlockType(BlockType blockType) throws IOException {
                    HFileBlock blk = this.nextBlock();
                    if (blk.getBlockType() != blockType) {
                        throw new IOException("Expected block of type " + (Object)((Object)blockType) + " but found " + (Object)((Object)blk.getBlockType()));
                    }
                    return blk;
                }
            };
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected int readAtOffset(FSDataInputStream istream, byte[] dest, int destOffset, int size, boolean peekIntoNextBlock, long fileOffset, boolean pread) throws IOException {
            block12: {
                if (peekIntoNextBlock && destOffset + size + this.hdrSize > dest.length) {
                    throw new IOException("Attempted to read " + size + " bytes and " + this.hdrSize + " bytes of next header into a " + dest.length + "-byte array at offset " + destOffset);
                }
                if (!pread && this.streamLock.tryLock()) {
                    try {
                        istream.seek(fileOffset);
                        long realOffset = istream.getPos();
                        if (realOffset != fileOffset) {
                            throw new IOException("Tried to seek to " + fileOffset + " to " + "read " + size + " bytes, but pos=" + realOffset + " after seek");
                        }
                        if (!peekIntoNextBlock) {
                            IOUtils.readFully((InputStream)istream, (byte[])dest, (int)destOffset, (int)size);
                            int n = -1;
                            return n;
                        }
                        if (!HFileBlock.readWithExtra((InputStream)istream, dest, destOffset, size, this.hdrSize)) {
                            int n = -1;
                            return n;
                        }
                        break block12;
                    }
                    finally {
                        this.streamLock.unlock();
                    }
                }
                int extraSize = peekIntoNextBlock ? this.hdrSize : 0;
                int ret = istream.read(fileOffset, dest, destOffset, size + extraSize);
                if (ret < size) {
                    throw new IOException("Positional read of " + size + " bytes " + "failed at offset " + fileOffset + " (returned " + ret + ")");
                }
                if (ret == size || ret < size + extraSize) {
                    return -1;
                }
            }
            assert (peekIntoNextBlock);
            return Bytes.toInt(dest, destOffset + size + 8) + this.hdrSize;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void decompress(byte[] dest, int destOffset, InputStream bufferedBoundedStream, int uncompressedSize) throws IOException {
            Decompressor decompressor = null;
            try {
                decompressor = this.compressAlgo.getDecompressor();
                InputStream is = this.compressAlgo.createDecompressionStream(bufferedBoundedStream, decompressor, 0);
                IOUtils.readFully((InputStream)is, (byte[])dest, (int)destOffset, (int)uncompressedSize);
                is.close();
            }
            finally {
                if (decompressor != null) {
                    this.compressAlgo.returnDecompressor(decompressor);
                }
            }
        }

        protected InputStream createBufferedBoundedStream(long offset, int size, boolean pread) {
            return new BufferedInputStream(new BoundedRangeFileInputStream(this.istream, offset, size, pread), Math.min(0x100000, size));
        }

        protected int getMinorVersion() {
            return this.minorVersion;
        }
    }

    public static interface FSReader {
        public HFileBlock readBlockData(long var1, long var3, int var5, boolean var6) throws IOException;

        public BlockIterator blockRange(long var1, long var3);
    }

    public static interface BlockIterator {
        public HFileBlock nextBlock() throws IOException;

        public HFileBlock nextBlockWithBlockType(BlockType var1) throws IOException;
    }

    public static interface BlockWritable {
        public BlockType getBlockType();

        public void writeToBlock(DataOutput var1) throws IOException;
    }

    public static class Writer {
        private State state = State.INIT;
        private final Compression.Algorithm compressAlgo;
        private final HFileDataBlockEncoder dataBlockEncoder;
        private ByteArrayOutputStream baosInMemory;
        private Compressor compressor;
        private CompressionOutputStream compressionStream;
        private ByteArrayOutputStream compressedByteStream;
        private BlockType blockType;
        private DataOutputStream userDataStream;
        private byte[] onDiskBytesWithHeader;
        private int onDiskDataSizeWithHeader;
        private byte[] onDiskChecksum;
        private byte[] uncompressedBytesWithHeader;
        private long startOffset;
        private long[] prevOffsetByType;
        private long prevOffset;
        private boolean includesMemstoreTS;
        private ChecksumType checksumType;
        private int bytesPerChecksum;
        private final int minorVersion;

        public Writer(Compression.Algorithm compressionAlgorithm, HFileDataBlockEncoder dataBlockEncoder, boolean includesMemstoreTS, int minorVersion, ChecksumType checksumType, int bytesPerChecksum) {
            this.minorVersion = minorVersion;
            this.compressAlgo = compressionAlgorithm == null ? Compression.Algorithm.NONE : compressionAlgorithm;
            this.dataBlockEncoder = dataBlockEncoder != null ? dataBlockEncoder : NoOpDataBlockEncoder.INSTANCE;
            this.baosInMemory = new ByteArrayOutputStream();
            if (this.compressAlgo != Compression.Algorithm.NONE) {
                this.compressor = compressionAlgorithm.getCompressor();
                this.compressedByteStream = new ByteArrayOutputStream();
                try {
                    this.compressionStream = compressionAlgorithm.createPlainCompressionStream(this.compressedByteStream, this.compressor);
                }
                catch (IOException e) {
                    throw new RuntimeException("Could not create compression stream for algorithm " + (Object)((Object)compressionAlgorithm), e);
                }
            }
            if (minorVersion > 0 && bytesPerChecksum < 33) {
                throw new RuntimeException("Unsupported value of bytesPerChecksum.  Minimum is 33 but the configured value is " + bytesPerChecksum);
            }
            this.prevOffsetByType = new long[BlockType.values().length];
            for (int i = 0; i < this.prevOffsetByType.length; ++i) {
                this.prevOffsetByType[i] = -1L;
            }
            this.includesMemstoreTS = includesMemstoreTS;
            this.checksumType = checksumType;
            this.bytesPerChecksum = bytesPerChecksum;
        }

        public DataOutputStream startWriting(BlockType newBlockType) throws IOException {
            if (this.state == State.BLOCK_READY && this.startOffset != -1L) {
                this.prevOffsetByType[this.blockType.getId()] = this.startOffset;
            }
            this.startOffset = -1L;
            this.blockType = newBlockType;
            this.baosInMemory.reset();
            this.baosInMemory.write(HFileBlock.getDummyHeaderForVersion(this.minorVersion));
            this.state = State.WRITING;
            this.userDataStream = new DataOutputStream(this.baosInMemory);
            return this.userDataStream;
        }

        DataOutputStream getUserDataStream() {
            this.expectState(State.WRITING);
            return this.userDataStream;
        }

        private void ensureBlockReady() throws IOException {
            Preconditions.checkState((this.state != State.INIT ? 1 : 0) != 0, (Object)("Unexpected state: " + (Object)((Object)this.state)));
            if (this.state == State.BLOCK_READY) {
                return;
            }
            this.finishBlock();
        }

        private void finishBlock() throws IOException {
            this.userDataStream.flush();
            this.uncompressedBytesWithHeader = this.baosInMemory.toByteArray();
            this.prevOffset = this.prevOffsetByType[this.blockType.getId()];
            this.state = State.BLOCK_READY;
            this.encodeDataBlockForDisk();
            this.doCompressionAndChecksumming();
        }

        private void doCompressionAndChecksumming() throws IOException {
            if (this.minorVersion <= 0) {
                this.version20compression();
            } else {
                this.version21ChecksumAndCompression();
            }
        }

        private void version20compression() throws IOException {
            this.onDiskChecksum = HConstants.EMPTY_BYTE_ARRAY;
            if (this.compressAlgo != Compression.Algorithm.NONE) {
                this.compressedByteStream.reset();
                this.compressedByteStream.write(DUMMY_HEADER_NO_CHECKSUM);
                this.compressionStream.resetState();
                this.compressionStream.write(this.uncompressedBytesWithHeader, HFileBlock.headerSize(this.minorVersion), this.uncompressedBytesWithHeader.length - HFileBlock.headerSize(this.minorVersion));
                this.compressionStream.flush();
                this.compressionStream.finish();
                this.onDiskDataSizeWithHeader = this.compressedByteStream.size();
                this.onDiskBytesWithHeader = this.compressedByteStream.toByteArray();
                this.put20Header(this.onDiskBytesWithHeader, 0, this.onDiskBytesWithHeader.length, this.uncompressedBytesWithHeader.length);
                this.put20Header(this.uncompressedBytesWithHeader, 0, this.onDiskBytesWithHeader.length + this.onDiskChecksum.length, this.uncompressedBytesWithHeader.length);
            } else {
                this.onDiskBytesWithHeader = this.uncompressedBytesWithHeader;
                this.onDiskDataSizeWithHeader = this.onDiskBytesWithHeader.length;
                this.put20Header(this.uncompressedBytesWithHeader, 0, this.onDiskBytesWithHeader.length, this.uncompressedBytesWithHeader.length);
            }
        }

        private void version21ChecksumAndCompression() throws IOException {
            if (this.compressAlgo != Compression.Algorithm.NONE) {
                this.compressedByteStream.reset();
                this.compressedByteStream.write(DUMMY_HEADER_WITH_CHECKSUM);
                this.compressionStream.resetState();
                this.compressionStream.write(this.uncompressedBytesWithHeader, HFileBlock.headerSize(this.minorVersion), this.uncompressedBytesWithHeader.length - HFileBlock.headerSize(this.minorVersion));
                this.compressionStream.flush();
                this.compressionStream.finish();
                this.onDiskDataSizeWithHeader = this.compressedByteStream.size();
                ChecksumUtil.reserveSpaceForChecksums(this.compressedByteStream, this.onDiskDataSizeWithHeader, this.bytesPerChecksum);
                this.onDiskBytesWithHeader = this.compressedByteStream.toByteArray();
                this.put21Header(this.onDiskBytesWithHeader, 0, this.onDiskBytesWithHeader.length, this.uncompressedBytesWithHeader.length, this.onDiskDataSizeWithHeader);
                ChecksumUtil.generateChecksums(this.onDiskBytesWithHeader, 0, this.onDiskDataSizeWithHeader, this.onDiskBytesWithHeader, this.onDiskDataSizeWithHeader, this.checksumType, this.bytesPerChecksum);
                this.onDiskChecksum = HConstants.EMPTY_BYTE_ARRAY;
                this.put21Header(this.uncompressedBytesWithHeader, 0, this.onDiskBytesWithHeader.length + this.onDiskChecksum.length, this.uncompressedBytesWithHeader.length, this.onDiskDataSizeWithHeader);
            } else {
                this.onDiskBytesWithHeader = this.uncompressedBytesWithHeader;
                this.onDiskDataSizeWithHeader = this.onDiskBytesWithHeader.length;
                int numBytes = (int)ChecksumUtil.numBytes(this.uncompressedBytesWithHeader.length, this.bytesPerChecksum);
                this.onDiskChecksum = new byte[numBytes];
                this.put21Header(this.uncompressedBytesWithHeader, 0, this.onDiskBytesWithHeader.length + this.onDiskChecksum.length, this.uncompressedBytesWithHeader.length, this.onDiskDataSizeWithHeader);
                ChecksumUtil.generateChecksums(this.uncompressedBytesWithHeader, 0, this.uncompressedBytesWithHeader.length, this.onDiskChecksum, 0, this.checksumType, this.bytesPerChecksum);
            }
        }

        private void encodeDataBlockForDisk() throws IOException {
            if (this.blockType != BlockType.DATA) {
                return;
            }
            ByteBuffer rawKeyValues = ByteBuffer.wrap(this.uncompressedBytesWithHeader, HFileBlock.headerSize(this.minorVersion), this.uncompressedBytesWithHeader.length - HFileBlock.headerSize(this.minorVersion)).slice();
            Pair<ByteBuffer, BlockType> encodingResult = this.dataBlockEncoder.beforeWriteToDisk(rawKeyValues, this.includesMemstoreTS, HFileBlock.getDummyHeaderForVersion(this.minorVersion));
            BlockType encodedBlockType = encodingResult.getSecond();
            if (encodedBlockType == BlockType.ENCODED_DATA) {
                this.uncompressedBytesWithHeader = encodingResult.getFirst().array();
                this.blockType = BlockType.ENCODED_DATA;
            } else {
                if (encodedBlockType != BlockType.DATA) {
                    throw new IOException("Unexpected block type coming out of data block encoder: " + (Object)((Object)encodedBlockType));
                }
                if (this.userDataStream.size() != this.uncompressedBytesWithHeader.length - HFileBlock.headerSize(this.minorVersion)) {
                    throw new IOException("Uncompressed size mismatch: " + this.userDataStream.size() + " vs. " + (this.uncompressedBytesWithHeader.length - HFileBlock.headerSize(this.minorVersion)));
                }
            }
        }

        private void put21Header(byte[] dest, int offset, int onDiskSize, int uncompressedSize, int onDiskDataSize) {
            offset = this.blockType.put(dest, offset);
            offset = Bytes.putInt(dest, offset, onDiskSize - 33);
            offset = Bytes.putInt(dest, offset, uncompressedSize - 33);
            offset = Bytes.putLong(dest, offset, this.prevOffset);
            offset = Bytes.putByte(dest, offset, this.checksumType.getCode());
            offset = Bytes.putInt(dest, offset, this.bytesPerChecksum);
            offset = Bytes.putInt(dest, offset, this.onDiskDataSizeWithHeader);
        }

        private void put20Header(byte[] dest, int offset, int onDiskSize, int uncompressedSize) {
            offset = this.blockType.put(dest, offset);
            offset = Bytes.putInt(dest, offset, onDiskSize - 24);
            offset = Bytes.putInt(dest, offset, uncompressedSize - 24);
            Bytes.putLong(dest, offset, this.prevOffset);
        }

        public void writeHeaderAndData(FSDataOutputStream out) throws IOException {
            long offset = out.getPos();
            if (this.startOffset != -1L && offset != this.startOffset) {
                throw new IOException("A " + (Object)((Object)this.blockType) + " block written to a " + "stream twice, first at offset " + this.startOffset + ", then at " + offset);
            }
            this.startOffset = offset;
            this.writeHeaderAndData((DataOutputStream)out);
        }

        private void writeHeaderAndData(DataOutputStream out) throws IOException {
            this.ensureBlockReady();
            out.write(this.onDiskBytesWithHeader);
            if (this.compressAlgo == Compression.Algorithm.NONE && this.minorVersion > 0) {
                if (this.onDiskChecksum == HConstants.EMPTY_BYTE_ARRAY) {
                    throw new IOException("A " + (Object)((Object)this.blockType) + " without compression should have checksums " + " stored separately.");
                }
                out.write(this.onDiskChecksum);
            }
        }

        byte[] getHeaderAndDataForTest() throws IOException {
            this.ensureBlockReady();
            if (this.compressAlgo == Compression.Algorithm.NONE) {
                if (this.onDiskChecksum == HConstants.EMPTY_BYTE_ARRAY) {
                    throw new IOException("A " + (Object)((Object)this.blockType) + " without compression should have checksums " + " stored separately.");
                }
                byte[] output = new byte[this.onDiskBytesWithHeader.length + this.onDiskChecksum.length];
                System.arraycopy(this.onDiskBytesWithHeader, 0, output, 0, this.onDiskBytesWithHeader.length);
                System.arraycopy(this.onDiskChecksum, 0, output, this.onDiskBytesWithHeader.length, this.onDiskChecksum.length);
                return output;
            }
            return this.onDiskBytesWithHeader;
        }

        public void releaseCompressor() {
            if (this.compressor != null) {
                this.compressAlgo.returnCompressor(this.compressor);
                this.compressor = null;
            }
        }

        int getOnDiskSizeWithoutHeader() {
            this.expectState(State.BLOCK_READY);
            return this.onDiskBytesWithHeader.length + this.onDiskChecksum.length - HFileBlock.headerSize(this.minorVersion);
        }

        int getOnDiskSizeWithHeader() {
            this.expectState(State.BLOCK_READY);
            return this.onDiskBytesWithHeader.length + this.onDiskChecksum.length;
        }

        int getUncompressedSizeWithoutHeader() {
            this.expectState(State.BLOCK_READY);
            return this.uncompressedBytesWithHeader.length - HFileBlock.headerSize(this.minorVersion);
        }

        int getUncompressedSizeWithHeader() {
            this.expectState(State.BLOCK_READY);
            return this.uncompressedBytesWithHeader.length;
        }

        public boolean isWriting() {
            return this.state == State.WRITING;
        }

        public int blockSizeWritten() {
            if (this.state != State.WRITING) {
                return 0;
            }
            return this.userDataStream.size();
        }

        ByteBuffer getUncompressedBufferWithHeader() {
            this.expectState(State.BLOCK_READY);
            return ByteBuffer.wrap(this.uncompressedBytesWithHeader);
        }

        private void expectState(State expectedState) {
            if (this.state != expectedState) {
                throw new IllegalStateException("Expected state: " + (Object)((Object)expectedState) + ", actual state: " + (Object)((Object)this.state));
            }
        }

        public void writeBlock(BlockWritable bw, FSDataOutputStream out) throws IOException {
            bw.writeToBlock(this.startWriting(bw.getBlockType()));
            this.writeHeaderAndData(out);
        }

        public HFileBlock getBlockForCaching() {
            return new HFileBlock(this.blockType, this.getOnDiskSizeWithoutHeader(), this.getUncompressedSizeWithoutHeader(), this.prevOffset, this.getUncompressedBufferWithHeader(), false, this.startOffset, this.includesMemstoreTS, this.minorVersion, 0, ChecksumType.NULL.getCode(), this.onDiskBytesWithHeader.length + this.onDiskChecksum.length);
        }

        private static enum State {
            INIT,
            WRITING,
            BLOCK_READY;

        }
    }
}

