//
//  utility.c
//
//  Author: Tai-Yi Huang (tyhuang)      June, 2000
//
//

#include "define.h"

CHAR PostLogFileName[] = "PostLog";

VOID PostLogFlushThread(LPVOID Pars) {

    while (TRUE) {

        AcquireRWLockShared(PostLogInfo.Lock);

        if (NULL != PostLogInfo.LogFileHandle) {

            if (STATUS_SUCCESS != FlushCachedData()) {

                ProcessErrorExit("PostLogFlushThread", "FlushCachedData fail");
            }
        }

        ReleaseRWLockShared(PostLogInfo.Lock);

        Sleep(POSTLOG_FLUSH_INTERVAL);
    }
}

VOID InitializePostLogInfo() {

    CHAR        Counter[COUNTER_LENGTH];
    PCHAR       BufPtr;

    PostLogInfo.LogFileHandle = NewPostLogFile();

    if (NULL == PostLogInfo.LogFileHandle) {

        ProcessErrorExit("InitializePostLogInfo", "Failed to open PostLog file");
    }

    // Allocate the first buffer

    if (NULL == PostLogInfo.CurrentBuffer) {

        PostLogInfo.FirstBuffer = PostLogInfo.CurrentBuffer = NewLogBuffer( 0 );

    } else {

        PostLogInfo.FirstBuffer = PostLogInfo.CurrentBuffer;

        PostLogInfo.CurrentBuffer->PostLogFileOffset = 0;
        PostLogInfo.CurrentBuffer->Next = NULL;
        PostLogInfo.CurrentBuffer->BytesUsed = 0;
    }

    if (NULL == PostLogInfo.FirstBuffer) {

        ProcessErrorExit("InitializePostLogInfo", "Allocate LogBuffer failed\n");
    }

    PostLogInfo.LogEntryCounter = 0;
    PostLogInfo.SeparatorChar = ' ';
    PostLogInfo.TerminatorChar = '\n';

    // Initialize the first buffer

    sprintf(Counter, "%10d", 0);
    BufPtr = (PostLogInfo.CurrentBuffer)->LogData;

    memcpy(BufPtr, Counter, COUNTER_LENGTH);
    BufPtr += COUNTER_LENGTH;
    *BufPtr = PostLogInfo.TerminatorChar;

    (PostLogInfo.CurrentBuffer)->BytesUsed += COUNTER_LENGTH + 1;
}

HANDLE NewPostLogFile() {

    HANDLE LogFileHandle;

    LogFileHandle = CreateFile(
                        PostLogFileName,
                        GENERIC_READ | GENERIC_WRITE,
                        FILE_SHARE_READ,
                        NULL,
                        CREATE_ALWAYS,
                        FILE_FLAG_RANDOM_ACCESS,
                        NULL
                        );

    if (INVALID_HANDLE_VALUE == LogFileHandle) {

        ProcessErrorExit("NewPostLogFile", "Fail to create postlog");
    }

    printf("Open Postlog\n");

    return LogFileHandle;
}

PLOG_BUFFER NewLogBuffer(unsigned __int64 FileOffset) {

    PLOG_BUFFER         LogBufPtr;
    PCHAR               DataBufPtr;

    LogBufPtr = (PLOG_BUFFER) malloc(sizeof(LOG_BUFFER));

    if (NULL == LogBufPtr) {

        ProcessErrorExit("NewLogBuffer", "malloc failed");
    }

    DataBufPtr = (PCHAR) VirtualAlloc(
                            NULL,
                            LOG_BUF_SIZE,
                            MEM_RESERVE | MEM_COMMIT,
                            PAGE_READWRITE
                            );

    if (NULL == DataBufPtr) {

        ProcessErrorExit("NewLogBuffer", "VirtualAlloc failed");
    }

    LogBufPtr->PostLogFileOffset = FileOffset;
    LogBufPtr->LogData = DataBufPtr;
    LogBufPtr->Next = NULL;
    LogBufPtr->BytesUsed = 0;

    return LogBufPtr;
}

DWORD FlushEntryCounter() {

    OVERLAPPED          theOverlapped;
    unsigned long       numBytesWritten;
    ULARGE_INTEGER      fileOffset;
    CHAR                Counter[COUNTER_LENGTH];
    PLOG_BUFFER         LogBuf = PostLogInfo.FirstBuffer;

    memcpy(&fileOffset, &(LogBuf->PostLogFileOffset), sizeof(fileOffset));

    sprintf(Counter, "%10d", PostLogInfo.LogEntryCounter);
    memcpy(LogBuf->LogData, Counter, COUNTER_LENGTH);

    memset(&theOverlapped, 0, sizeof(theOverlapped));

    theOverlapped.Offset = fileOffset.LowPart;
    theOverlapped.OffsetHigh = fileOffset.HighPart;

    if (!WriteFile(
            PostLogInfo.LogFileHandle,
            LogBuf->LogData,
            COUNTER_LENGTH,
            &numBytesWritten,
            &theOverlapped)) {

        ProcessErrorExit("FlushEntryCounter", "WriteFile fail");
    }

    if (COUNTER_LENGTH != numBytesWritten) {

        ProcessErrorExit("FlushEntryCounter", "WriteBytes less");
    }

    return STATUS_SUCCESS;
}
    


DWORD FlushLogBuffer(PLOG_BUFFER LogBuf) {

    OVERLAPPED theOverlapped;
    unsigned long numBytesWritten, numBytesToWrite;
    ULARGE_INTEGER fileOffset;

    memcpy(&fileOffset, &(LogBuf->PostLogFileOffset), sizeof(fileOffset));

    memset(&theOverlapped, 0, sizeof(theOverlapped));

    theOverlapped.Offset = fileOffset.LowPart;
    theOverlapped.OffsetHigh = fileOffset.HighPart;

    numBytesToWrite = LogBuf->BytesUsed;

    if (!WriteFile(
            PostLogInfo.LogFileHandle,
            LogBuf->LogData,
            numBytesToWrite,
            &numBytesWritten,
            &theOverlapped
            )) {

        ProcessErrorExit("FlushLogBuffer", "WriteFile fail");
    }

    if (numBytesToWrite != numBytesWritten) {

        ProcessErrorExit("FlushLogBuffer", "WriteBytes less");
    }

    return STATUS_SUCCESS;
}


DWORD FlushDirtyBuffers() {

    DWORD           Status = STATUS_SUCCESS;
    PLOG_BUFFER     DirtyList, OrderedList, Temp;

    if (NULL == PostLogInfo.DirtyBufferList) {

        return Status;
    }

    // Otherwise, we need to flush the dirty buffers

    while (TRUE) {

        DirtyList = PostLogInfo.DirtyBufferList;

        if (DirtyList == InterlockedCompareExchangePointerPrivate((VOID **) &(PostLogInfo.DirtyBufferList), NULL, DirtyList)) {
            break;
        }
    }

    // dirty buffers are now in the private list. Reverse

    Temp = NULL;
    OrderedList = NULL;

    while (DirtyList) {

        Temp = DirtyList->Next;
        DirtyList->Next = OrderedList;
        OrderedList = DirtyList;
        DirtyList = Temp;
    }

    //
    //  TYHUANG: check if the first buffer is in the dirty list, if so
    //  flush the buffer and take it out from the dirty list such that
    //  it will not be returned to the free buffer list
    //

    if (OrderedList == PostLogInfo.FirstBuffer) {

        if (FlushLogBuffer(OrderedList) != STATUS_SUCCESS)
            return STATUS_ERROR;

        OrderedList = OrderedList->Next;
    }
    else { // search through the rest of dirty buffer

        DirtyList = OrderedList;
        Temp = DirtyList->Next;

        while (Temp) {

            if (Temp == PostLogInfo.FirstBuffer) {

                if (FlushLogBuffer(Temp) != STATUS_SUCCESS)
                    return STATUS_ERROR;

                DirtyList->Next = Temp->Next;
                break;
            }

            DirtyList = Temp;
            Temp = DirtyList->Next;
        }
    }


    //
    // Now flush. Still need to figure out a good way to handle failures
    //

    DirtyList = OrderedList;
    while (DirtyList) {

        Temp = DirtyList;
        DirtyList = DirtyList->Next;

        if (FlushLogBuffer(Temp) != STATUS_SUCCESS) {

            Status = STATUS_ERROR;
        }
    }

    //
    // I suspect the Interlocked is not necessary here, but for now, it's here.
    //

    while (Temp) {

        Temp->Next = PostLogInfo.FreeBufferList;

        if ((Temp->Next) == InterlockedCompareExchangePointerPrivate((VOID **) &(PostLogInfo.FreeBufferList), OrderedList, Temp->Next)) {

            break;
        }
    }

    return Status;

}

DWORD FlushCachedData() {

    DWORD        Status = STATUS_SUCCESS;

    // First, write dirty buffer

    if (STATUS_SUCCESS != FlushDirtyBuffers()) {

        ProcessErrorExit("FlushCachedData", "FlushDirtyBuffers fail");

        Status = STATUS_ERROR;
    }

    // Second, flush the entry counter 

    if (PostLogInfo.FirstBuffer != PostLogInfo.CurrentBuffer) {

        if (STATUS_SUCCESS != FlushEntryCounter()) {

            ProcessErrorExit("FlushCachedData", "FlushEntryCounter fail");

            Status = STATUS_ERROR;
        }
    }

    if (!FlushFileBuffers(PostLogInfo.LogFileHandle)) {

        ProcessErrorExit("FlushCachedData", "FlushFileBuffers fail");

        Status = STATUS_ERROR;
    }

    return Status;
}

DWORD FlushCachedDataExclusive() {

    DWORD        Status = STATUS_SUCCESS;

    // First, flush the dirty buffer

    if (STATUS_SUCCESS != FlushDirtyBuffers()) {

        ProcessErrorExit("FlushCachedDataExclusive", "FlushDirtyBuffers fail");

        Status = STATUS_ERROR;
    }

    // Second, flush the current buffer

    if (PostLogInfo.CurrentBuffer && (FlushLogBuffer(PostLogInfo.CurrentBuffer) != STATUS_SUCCESS)) {

        ProcessErrorExit("FlushCachedDataExclusive", "FlushLogBuffer fail");

        Status = STATUS_ERROR;
    }

    // Third, flush the first buffer

    if (PostLogInfo.FirstBuffer && (FlushEntryCounter() != STATUS_SUCCESS)) {

        ProcessErrorExit("FlushCachedDataExclusive", "FlushEntryCounter fail");

        Status = STATUS_ERROR;
    }

    return Status;
}

VOID ConvertNumberToStringWithSpace(PCHAR StrPtr, ULONG Number, ULONG DigitFields) {

    BOOLEAN     RecordingZero = FALSE;
    ULONG       DivNumber = 1;
    ULONG       i, A;

    while (DigitFields > 10) {

        *StrPtr++ = ' ';
        DigitFields--;
    }

    for (i=0;i<(DigitFields-1);i++) DivNumber *= 10;

    for (i=0;i<DigitFields;i++) {

        A = Number / DivNumber;

        if (A == 0) {

            if (RecordingZero || DivNumber == 1) {

                *StrPtr++ = (CHAR)('0');

            } else {

                *StrPtr++ = ' ';
            }

        } else {

            *StrPtr++ = (CHAR)('0' + A);
            RecordingZero = TRUE;
        }

        Number -= DivNumber * A;
        DivNumber /= 10;
    }
}


VOID CopyLogRecord(PCHAR BufPtr, PLOG_RECORD Record) {

    PCHAR   StrPtr = BufPtr;
    
    ConvertNumberToStringWithSpace(StrPtr, Record->RecordNum, 10);
    StrPtr += 10;

    ConvertNumberToStringWithSpace(StrPtr, Record->TimeStamp, 11);
    StrPtr += 11;

    ConvertNumberToStringWithSpace(StrPtr, Record->Pid, 11);
    StrPtr += 11;

    ConvertNumberToStringWithSpace(StrPtr, Record->DirNum, 6);
    StrPtr += 6;

    ConvertNumberToStringWithSpace(StrPtr, Record->ClassNum, 3);
    StrPtr += 3;

    ConvertNumberToStringWithSpace(StrPtr, Record->FileNum, 3);
    StrPtr += 3;

    ConvertNumberToStringWithSpace(StrPtr, Record->ClientNum, 11);
    StrPtr += 11;

    memcpy(StrPtr++, " ", 1);

    memcpy(StrPtr, Record->FileName, strlen(Record->FileName));
    StrPtr += strlen(Record->FileName);

    ConvertNumberToStringWithSpace(StrPtr, Record->Pid, 11);
    StrPtr += 11;

    ConvertNumberToStringWithSpace(StrPtr, Record->MyCookie, 11);
    StrPtr += 11;

    memcpy(StrPtr++, "\n", 1);

    // LOG_RECORD_LENGTH = 106 = StrPtr - BufPtr;
}


DWORD WritePostLogShared(PLOG_RECORD Record) {

    ULONG   StartPos;
    PCHAR   StartAddr;

    AcquireRWLockShared(PostLogInfo.Lock);

    if (PostLogInfo.CurrentBuffer) {

        AcquireRWLockExclusive(PostLogInfo.ShLock);

        StartPos = InterlockedExchangeAdd(&(PostLogInfo.CurrentBuffer->BytesUsed), LOG_RECORD_LENGTH);

        if (StartPos + LOG_RECORD_LENGTH <= LOG_BUF_SIZE) {

            StartAddr = PostLogInfo.CurrentBuffer->LogData + StartPos;

            Record->RecordNum = InterlockedIncrement(&(PostLogInfo.LogEntryCounter));

            ReleaseRWLockExclusive(PostLogInfo.ShLock);

            CopyLogRecord(StartAddr, Record);

            ReleaseRWLockShared(PostLogInfo.Lock);

            return STATUS_SUCCESS;

        } else {

            InterlockedExchangeAdd(&(PostLogInfo.CurrentBuffer->BytesUsed), (LONG) -1 * LOG_RECORD_LENGTH);

            ReleaseRWLockExclusive(PostLogInfo.ShLock);

            ReleaseRWLockShared(PostLogInfo.Lock);

            return STATUS_ERROR;
        }

    } else {

        ReleaseRWLockShared(PostLogInfo.Lock);

        ProcessErrorExit("WritePostLogShared", "CurrentBuffer is NULL");

        return STATUS_ERROR;
    }
}


DWORD WritePostLogExclusive(PLOG_RECORD Record) {

    CHAR                RecordString[160];
    PLOG_BUFFER         cbpt;
    ULONG               residualBytes, bytesToWrite;

    //
    // Grab the lock and get to work
    //

    AcquireRWLockExclusive(PostLogInfo.Lock);

    cbpt = PostLogInfo.CurrentBuffer;

    if (!cbpt)
    {

        ProcessErrorExit("WritePostLogExclusive", "current buffer is null");

        ReleaseRWLockExclusive(PostLogInfo.Lock);
        return (STATUS_ERROR);
    }

    //
    // copy log record to the record buffer
    //

    Record->RecordNum =  InterlockedIncrement((PULONG)&PostLogInfo.LogEntryCounter);

    CopyLogRecord(RecordString, Record);

    //
    // Note that we have the lock exclusive
    // so we won't need to worry about anything changing
    // First we check that we're not going to overflow the file
    //


    if (cbpt->PostLogFileOffset + cbpt->BytesUsed + LOG_RECORD_LENGTH > LOG_FILE_SIZE)
    {
        //
        //  this should not happen in ISAPI. Just return error
        //

        ProcessErrorExit("WriteLogExclusive", "PostLog is full");

        ReleaseRWLockExclusive(PostLogInfo.Lock);
        return (STATUS_ERROR);
    }

    //
    // Here, we know we will not over-run the log file
    //  (but can still over-run the buffer)
    //

    if ((cbpt->BytesUsed + LOG_RECORD_LENGTH) <= LOG_BUF_SIZE) {

        memcpy(&(cbpt->LogData[cbpt->BytesUsed]), RecordString, LOG_RECORD_LENGTH);
        cbpt->BytesUsed += LOG_RECORD_LENGTH;

    } else {

        //
        // copy what we can. No point checking if bytesToWrite is 0. Note that the separator
        // has to end up in the new buffer because we know we are at least one short on space
        //

        bytesToWrite = LOG_BUF_SIZE - cbpt->BytesUsed;
        memcpy(&(cbpt->LogData[cbpt->BytesUsed]), RecordString, bytesToWrite);
        residualBytes = LOG_RECORD_LENGTH - bytesToWrite;
        cbpt->BytesUsed = LOG_BUF_SIZE;

        //
        // Put the current buffer on the dirty list. We have the lock exclusive.
        //

        PostLogInfo.CurrentBuffer->Next = PostLogInfo.DirtyBufferList;
        PostLogInfo.DirtyBufferList = PostLogInfo.CurrentBuffer;

        //
        // Now get a buffer from the free list
        //

        if (PostLogInfo.FreeBufferList) {

            PostLogInfo.CurrentBuffer = PostLogInfo.FreeBufferList;
            PostLogInfo.FreeBufferList = PostLogInfo.CurrentBuffer->Next;

            // Initialize the current buffer

            PostLogInfo.CurrentBuffer->PostLogFileOffset = PostLogInfo.DirtyBufferList->PostLogFileOffset + LOG_BUF_SIZE;
            PostLogInfo.CurrentBuffer->Next = NULL;
            PostLogInfo.CurrentBuffer->BytesUsed = 0;

        } else {

            PostLogInfo.CurrentBuffer = NewLogBuffer(PostLogInfo.CurrentBuffer->PostLogFileOffset + LOG_BUF_SIZE);

            if (NULL == PostLogInfo.CurrentBuffer) {

                ProcessErrorExit("WriteLogExclusive", "NewLogBuffer fail");

                ReleaseRWLockExclusive(PostLogInfo.Lock);
                return (STATUS_ERROR);
            }
        }

        //
        // one way or the other, we made some space in (logInfo.currentBuffer)
        // write the rest of this entry into the brand new buffer
        //

        cbpt = PostLogInfo.CurrentBuffer;

        memcpy(&(cbpt->LogData[cbpt->BytesUsed]),
                    RecordString + bytesToWrite,
                    residualBytes
                    );
        cbpt->BytesUsed = residualBytes;

    } // if overflow

    //
    // we're done logging
    //

    ReleaseRWLockExclusive(PostLogInfo.Lock);

    return STATUS_SUCCESS;
}
